<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\ProductRequest;
use App\Models\Product;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth; // Ajout de la façade Auth
use Illuminate\Support\Facades\Notification;
use App\Notifications\NewProductRequestNotification;

class ProductRequestController extends Controller
{
    /**
     * Lister les demandes en fonction du rôle de l'utilisateur.
     */
    public function index()
    {
        $user = Auth::user();

        // On prépare la requête de base avec la relation produit
        $query = ProductRequest::with('product');

        // FILTRAGE : Si l'utilisateur est un chef de projet (et pas admin)
        // Assure-toi que la méthode hasRole() correspond à ton système de rôles (ex: Spatie)
        if ($user && $user->hasRole('chef_projet') && !$user->hasRole('admin')) {
            // On limite strictement à SES demandes via son ID
            $query->where('user_id', $user->id);
        }

        // TRI : Les demandes 'en_attente' en premier, puis par date décroissante
        $requests = $query->orderByRaw("CASE WHEN status IN ('en_attente', 'pending') THEN 1 ELSE 2 END")
            ->orderBy('created_at', 'desc')
            ->get();

        return response()->json($requests);
    }

    /**
     * Enregistrer une nouvelle demande de produit.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'product_id'    => 'required|exists:products,id',
            'quantity'      => 'required|integer|min:1',
            'employee_name' => 'required|string',
            'service_name'  => 'required|string',
            'site_location' => 'required|string',
            'created_by'    => 'nullable|string',
        ]);

        $product = Product::findOrFail($validated['product_id']);

        $productRequest = ProductRequest::create([
            'product_id'    => $validated['product_id'],
            'quantity'      => $validated['quantity'],
            'employee_name' => $validated['employee_name'],
            'service_name'  => $validated['service_name'],
            'site_location' => $validated['site_location'],
            // On garde le nom pour l'affichage si besoin
            'created_by'    => Auth::user()->name ?? 'Système',
            // NOUVEAU : On enregistre l'ID de l'utilisateur pour le filtrage
            'user_id'       => Auth::id(),
            'unit'          => $product->unit,
            'status'        => 'en_attente',
        ]);

        return response()->json($productRequest, 201);
    }

    /**
     * Approuver ou rejeter une demande avec gestion automatique du stock.
     */
    public function updateStatus(Request $request, $id)
    {
        $validated = $request->validate([
            'status' => 'required|in:approved,rejected',
        ]);

        return DB::transaction(function () use ($validated, $id) {
            $productRequest = ProductRequest::findOrFail($id);

            if ($productRequest->status !== 'en_attente') {
                return response()->json(['message' => 'Cette demande a déjà été traitée.'], 400);
            }

            if ($validated['status'] === 'approved') {
                $product = Product::findOrFail($productRequest->product_id);

                if ($product->quantity < $productRequest->quantity) {
                    return response()->json(['message' => 'Stock insuffisant pour approuver cette demande.'], 400);
                }

                $product->decrement('quantity', $productRequest->quantity);
            }

            $productRequest->update(['status' => $validated['status']]);

            return response()->json([
                'message' => 'Statut mis à jour avec succès.',
                'data' => $productRequest->load('product')
            ]);
        });
    }
}
