<?php

namespace App\Http\Controllers;

use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rule;

class UserController extends Controller
{
    public function store(Request $request)
    {
        // Seul l'admin devrait pouvoir appeler cette route (vérifié par middleware)
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users',
            'password' => 'required|string|min:6',
            'role' => ['required', Rule::in(['admin', 'magasinier', 'chef_projet'])],
        ]);

        $user = User::create([
            'name' => $validated['name'],
            'email' => $validated['email'],
            'password' => Hash::make($validated['password']),
            'role' => $validated['role'],
        ]);



        return response()->json([
            'message' => 'Utilisateur créé avec succès',
            'user' => $user
        ], 201);
    }
    // Liste des utilisateurs
public function index() {
    return response()->json(User::all());
}

// Suppression d'un utilisateur
public function destroy($id) {
    $user = User::findOrFail($id);

    // Sécurité : Empêcher l'admin de se supprimer lui-même par erreur
    if (auth()->id() == $id) {
        return response()->json(['message' => 'Action interdite'], 403);
    }

    $user->delete();
    return response()->json(['message' => 'Utilisateur supprimé']);
}

public function updateProfile(Request $request) {
    $user = $request->user();

    $request->validate([
        'name' => 'required|string|max:255',
        'email' => 'required|email|unique:users,email,' . $user->id,
        'phone' => 'nullable|string|max:20', // Validation du contact
        'password' => 'nullable|min:6|confirmed',
    ]);

    $user->name = $request->name;
    $user->email = $request->email;
    $user->phone = $request->phone; // On sauvegarde le contact

    if ($request->filled('password')) {
        $user->password = Hash::make($request->password);
    }

    $user->save();

    return response()->json(['message' => 'Profil mis à jour', 'user' => $user]);
}

public function update(Request $request, $id) {
    $user = User::findOrFail($id);

    $validated = $request->validate([
        'name' => 'required|string|max:255',
        'email' => 'required|email|unique:users,email,'.$id,
        'role' => ['required', Rule::in(['admin', 'magasinier', 'chef_projet'])],
        'phone' => 'nullable|string|max:20',
        'password' => 'nullable|min:6' // Optionnel ici
    ]);

    if ($request->filled('password')) {
        $validated['password'] = Hash::make($request->password);
    } else {
        unset($validated['password']); // On ne touche pas au mot de passe s'il est vide
    }

    $user->update($validated);

    return response()->json(['message' => 'Utilisateur mis à jour', 'user' => $user]);
}

}
